// assets/js/admin.js - Admin Panel JavaScript
// جاوااسکریپت پنل مدیریت

class AdminManager {
    constructor() {
        this.currentPage = 'dashboard';
        this.charts = {};
        this.tables = {};
        this.filters = {};
        this.isLoading = false;
        
        this.init();
    }
    
    init() {
        this.setupEventListeners();
        this.loadDashboardData();
        this.initializeCharts();
        this.setupDataTables();
        this.startRealTimeUpdates();
    }
    
    // Setup event listeners
    setupEventListeners() {
        // Sidebar navigation
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('admin-nav-link')) {
                e.preventDefault();
                this.navigateToPage(e.target.dataset.page);
            }
            
            // User actions
            if (e.target.classList.contains('ban-user-btn')) {
                this.banUser(e.target.dataset.userId);
            }
            
            if (e.target.classList.contains('unban-user-btn')) {
                this.unbanUser(e.target.dataset.userId);
            }
            
            if (e.target.classList.contains('verify-user-btn')) {
                this.verifyUser(e.target.dataset.userId);
            }
            
            // Transaction actions
            if (e.target.classList.contains('approve-transaction-btn')) {
                this.approveTransaction(e.target.dataset.transactionId);
            }
            
            if (e.target.classList.contains('reject-transaction-btn')) {
                this.rejectTransaction(e.target.dataset.transactionId);
            }
            
            // Support ticket actions
            if (e.target.classList.contains('reply-ticket-btn')) {
                this.showTicketReplyModal(e.target.dataset.ticketId);
            }
            
            if (e.target.classList.contains('close-ticket-btn')) {
                this.closeTicket(e.target.dataset.ticketId);
            }
            
            // Settings actions
            if (e.target.classList.contains('save-settings-btn')) {
                this.saveSettings();
            }
            
            // Export actions
            if (e.target.classList.contains('export-btn')) {
                this.exportData(e.target.dataset.type);
            }
        });
        
        // Form submissions
        document.addEventListener('submit', (e) => {
            if (e.target.classList.contains('admin-form')) {
                e.preventDefault();
                this.handleFormSubmission(e.target);
            }
        });
        
        // Filter changes
        document.addEventListener('change', (e) => {
            if (e.target.classList.contains('admin-filter')) {
                this.applyFilters();
            }
        });
        
        // Search functionality
        document.addEventListener('input', (e) => {
            if (e.target.classList.contains('admin-search')) {
                this.debounce(() => {
                    this.performSearch(e.target.value, e.target.dataset.searchType);
                }, 500)();
            }
        });
    }
    
    // Navigate to different admin pages
    navigateToPage(page) {
        // Update active navigation
        document.querySelectorAll('.admin-nav-link').forEach(link => {
            link.classList.remove('active');
        });
        
        document.querySelector(`[data-page="${page}"]`).classList.add('active');
        
        // Hide all page contents
        document.querySelectorAll('.admin-page-content').forEach(content => {
            content.style.display = 'none';
        });
        
        // Show selected page
        const pageContent = document.querySelector(`#admin-${page}`);
        if (pageContent) {
            pageContent.style.display = 'block';
            this.currentPage = page;
            
            // Load page-specific data
            this.loadPageData(page);
        }
    }
    
    // Load dashboard data
    async loadDashboardData() {
        try {
            this.showLoading();
            
            const response = await fetch('/admin/api/dashboard.php');
            const data = await response.json();
            
            if (data.success) {
                this.updateDashboardStats(data.data.stats);
                this.updateDashboardCharts(data.data.charts);
                this.updateRecentActivity(data.data.recent_activity);
            } else {
                throw new Error(data.error || 'خطا در بارگذاری داده‌های داشبورد');
            }
        } catch (error) {
            console.error('Error loading dashboard data:', error);
            this.showNotification(error.message, 'error');
        } finally {
            this.hideLoading();
        }
    }
    
    // Update dashboard statistics
    updateDashboardStats(stats) {
        Object.keys(stats).forEach(key => {
            const element = document.querySelector(`[data-stat="${key}"]`);
            if (element) {
                const valueElement = element.querySelector('.stat-value');
                const changeElement = element.querySelector('.stat-change');
                
                if (valueElement) {
                    this.animateNumber(valueElement, stats[key].value);
                }
                
                if (changeElement && stats[key].change !== undefined) {
                    const change = stats[key].change;
                    changeElement.textContent = `${change > 0 ? '+' : ''}${change}%`;
                    changeElement.className = `stat-change ${change >= 0 ? 'positive' : 'negative'}`;
                }
            }
        });
    }
    
    // User management actions
    async banUser(userId) {
        if (!confirm('آیا از مسدود کردن این کاربر اطمینان دارید؟')) {
            return;
        }
        
        try {
            const response = await fetch('/admin/api/users.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'ban',
                    user_id: userId
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showNotification('کاربر با موفقیت مسدود شد', 'success');
                this.loadUsersData();
            } else {
                throw new Error(result.error || 'خطا در مسدود کردن کاربر');
            }
        } catch (error) {
            console.error('Error banning user:', error);
            this.showNotification(error.message, 'error');
        }
    }
    
    async unbanUser(userId) {
        try {
            const response = await fetch('/admin/api/users.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'unban',
                    user_id: userId
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showNotification('کاربر با موفقیت فعال شد', 'success');
                this.loadUsersData();
            } else {
                throw new Error(result.error || 'خطا در فعال کردن کاربر');
            }
        } catch (error) {
            console.error('Error unbanning user:', error);
            this.showNotification(error.message, 'error');
        }
    }
    
    // Initialize charts
    initializeCharts() {
        // Revenue chart
        const revenueCtx = document.getElementById('revenueChart');
        if (revenueCtx) {
            this.charts.revenue = new Chart(revenueCtx, {
                type: 'line',
                data: {
                    labels: [],
                    datasets: [{
                        label: 'درآمد',
                        data: [],
                        borderColor: '#007bff',
                        backgroundColor: 'rgba(0, 123, 255, 0.1)',
                        tension: 0.4
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true
                        }
                    }
                }
            });
        }
        
        // Users chart
        const usersCtx = document.getElementById('usersChart');
        if (usersCtx) {
            this.charts.users = new Chart(usersCtx, {
                type: 'bar',
                data: {
                    labels: [],
                    datasets: [{
                        label: 'کاربران جدید',
                        data: [],
                        backgroundColor: '#28a745'
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true
                        }
                    }
                }
            });
        }
    }
    
    // Utility functions
    animateNumber(element, targetValue) {
        const startValue = parseInt(element.textContent) || 0;
        const duration = 1000;
        const startTime = Date.now();
        
        const animate = () => {
            const elapsed = Date.now() - startTime;
            const progress = Math.min(elapsed / duration, 1);
            
            const currentValue = Math.floor(startValue + (targetValue - startValue) * progress);
            element.textContent = currentValue.toLocaleString();
            
            if (progress < 1) {
                requestAnimationFrame(animate);
            }
        };
        
        animate();
    }
    
    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
    
    showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `alert alert-${type} notification fade-in`;
        notification.innerHTML = `
            <div class="notification-content">
                <i class="fas fa-${this.getNotificationIcon(type)}"></i>
                <span>${message}</span>
            </div>
            <button class="notification-close">&times;</button>
        `;
        
        let container = document.querySelector('.notifications-container');
        if (!container) {
            container = document.createElement('div');
            container.className = 'notifications-container';
            document.body.appendChild(container);
        }
        
        container.appendChild(notification);
        
        notification.querySelector('.notification-close').addEventListener('click', () => {
            this.removeNotification(notification);
        });
        
        setTimeout(() => {
            this.removeNotification(notification);
        }, 5000);
    }
    
    removeNotification(notification) {
        if (notification.parentNode) {
            notification.classList.add('fade-out');
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }
    }
    
    getNotificationIcon(type) {
        const icons = {
            'success': 'check-circle',
            'error': 'exclamation-circle',
            'warning': 'exclamation-triangle',
            'info': 'info-circle'
        };
        
        return icons[type] || 'info-circle';
    }
    
    showLoading(message = 'در حال بارگذاری...') {
        let loader = document.querySelector('.admin-loading-overlay');
        if (!loader) {
            loader = document.createElement('div');
            loader.className = 'admin-loading-overlay';
            loader.innerHTML = `
                <div class="loading-content">
                    <div class="loading-spinner"></div>
                    <div class="loading-message">${message}</div>
                </div>
            `;
            document.body.appendChild(loader);
        }
        
        loader.style.display = 'flex';
        this.isLoading = true;
    }
    
    hideLoading() {
        const loader = document.querySelector('.admin-loading-overlay');
        if (loader) {
            loader.style.display = 'none';
        }
        this.isLoading = false;
    }
    
    startRealTimeUpdates() {
        setInterval(() => {
            if (this.currentPage === 'dashboard') {
                this.loadDashboardData();
            }
        }, 30000);
    }
    
    setupDataTables() {
        // Initialize DataTables for admin tables
        if (typeof $ !== 'undefined' && $.fn.DataTable) {
            $('.admin-table').DataTable({
                responsive: true,
                language: {
                    url: '/assets/js/datatables-fa.json'
                }
            });
        }
    }
}

// Initialize admin manager when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    if (document.body.classList.contains('admin-page')) {
        window.adminManager = new AdminManager();
    }
});
