<?php
// pages/sportsbook.php - Sportsbook Page
// صفحه شرط‌بندی ورزشی

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Page settings
$pageTitle = 'Sportsbook - Sports Betting';
$bodyClass = 'sportsbook-page';
$breadcrumbs = [
    ['title' => 'Sportsbook']
];

// Get filters from URL
$selectedSport = $_GET['sport'] ?? 'all';
$selectedLeague = $_GET['league'] ?? 'all';
$timeFilter = $_GET['time'] ?? 'today';
$searchQuery = $_GET['search'] ?? '';
$viewMode = $_GET['view'] ?? 'grid';

// Get sports data
$sportsData = getSportsData();
$featuredEvents = getFeaturedEvents(6);
$liveEvents = getLiveEvents(10);
$upcomingEvents = getUpcomingEvents($selectedSport, $selectedLeague, $timeFilter, $searchQuery, 50);

// Get leagues for selected sport
$leagues = [];
if ($selectedSport !== 'all') {
    $leagues = fetchAll(
        "SELECT l.* FROM leagues l 
         JOIN sports s ON l.sport_id = s.id 
         WHERE s.slug = ? AND l.active = 1 
         ORDER BY l.name",
        [$selectedSport]
    );
}

// Additional scripts
$additionalScripts = [
    '/assets/js/betting.js',
    '/assets/js/sportsbook.js'
];

include '../includes/header.php';
?>

<!-- Hero Section -->
<section class="sportsbook-hero">
    <div class="container">
        <div class="hero-content">
            <div class="hero-text">
                <h1 class="hero-title">
                    <i class="fas fa-futbol"></i>
                    Sports Betting
                </h1>
                <p class="hero-description">
                    Bet on your favorite sports with the best odds and instant crypto payouts. 
                    Live betting, pre-match markets, and exclusive promotions available.
                </p>
                <div class="hero-stats">
                    <div class="stat-item">
                        <div class="stat-number"><?php echo count($liveEvents); ?></div>
                        <div class="stat-label">Live Events</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo count($upcomingEvents); ?></div>
                        <div class="stat-label">Upcoming</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo count($sportsData); ?></div>
                        <div class="stat-label">Sports</div>
                    </div>
                </div>
            </div>
            <div class="hero-image">
                <div class="floating-cards">
                    <div class="betting-card">
                        <div class="card-header">
                            <span class="live-indicator">LIVE</span>
                            <span class="match-time">67'</span>
                        </div>
                        <div class="match-info">
                            <div class="team">
                                <img src="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'assets/images/teams/barcelona.png" alt="Barcelona">
                                <span>Barcelona</span>
                                <span class="score">2</span>
                            </div>
                            <div class="vs">VS</div>
                            <div class="team">
                                <img src="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'assets/images/teams/real-madrid.png" alt="Real Madrid">
                                <span>Real Madrid</span>
                                <span class="score">1</span>
                            </div>
                        </div>
                        <div class="odds-row">
                            <button class="odds-btn">1.85</button>
                            <button class="odds-btn">3.20</button>
                            <button class="odds-btn">4.50</button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Sports Navigation -->
<section class="sports-nav-section">
    <div class="container">
        <div class="sports-nav">
            <div class="sports-tabs">
                <a href="?sport=all" class="sport-tab <?php echo $selectedSport === 'all' ? 'active' : ''; ?>">
                    <i class="fas fa-globe"></i>
                    <span>All Sports</span>
                    <span class="event-count"><?php echo array_sum(array_column($sportsData, 'total_events')); ?></span>
                </a>
                <?php foreach ($sportsData as $sport): ?>
                <a href="?sport=<?php echo $sport['slug']; ?>" 
                   class="sport-tab <?php echo $selectedSport === $sport['slug'] ? 'active' : ''; ?>">
                    <i class="<?php echo $sport['icon']; ?>"></i>
                    <span><?php echo $sport['name']; ?></span>
                    <span class="event-count"><?php echo $sport['total_events']; ?></span>
                    <?php if ($sport['live_count'] > 0): ?>
                    <span class="live-badge"><?php echo $sport['live_count']; ?></span>
                    <?php endif; ?>
                </a>
                <?php endforeach; ?>
            </div>
            
            <div class="nav-actions">
                <button class="live-events-btn" onclick="showLiveEvents()">
                    <i class="fas fa-broadcast-tower"></i>
                    <span>Live Events</span>
                    <span class="live-count"><?php echo count($liveEvents); ?></span>
                </button>
                <button class="favorites-btn" onclick="showFavorites()">
                    <i class="fas fa-star"></i>
                    <span>Favorites</span>
                </button>
            </div>
        </div>
    </div>
</section>

<!-- Live Events Ticker -->
<?php if (!empty($liveEvents)): ?>
<section class="live-ticker">
    <div class="container">
        <div class="ticker-header">
            <div class="ticker-title">
                <i class="fas fa-circle live-dot"></i>
                <span>Live Now</span>
            </div>
            <div class="ticker-controls">
                <button class="ticker-prev" onclick="scrollTicker('prev')">
                    <i class="fas fa-chevron-left"></i>
                </button>
                <button class="ticker-next" onclick="scrollTicker('next')">
                    <i class="fas fa-chevron-right"></i>
                </button>
            </div>
        </div>
        <div class="ticker-content" id="liveTicker">
            <?php foreach ($liveEvents as $event): ?>
            <div class="ticker-item" data-event-id="<?php echo $event['id']; ?>">
                <div class="match-info">
                    <div class="teams">
                        <span class="home-team"><?php echo $event['home_team']; ?></span>
                        <span class="score"><?php echo $event['home_score']; ?>-<?php echo $event['away_score']; ?></span>
                        <span class="away-team"><?php echo $event['away_team']; ?></span>
                    </div>
                    <div class="match-status">
                        <span class="time"><?php echo $event['match_time']; ?>'</span>
                        <span class="sport"><?php echo $event['sport_name']; ?></span>
                    </div>
                </div>
                <div class="quick-odds">
                    <?php if ($event['home_odds']): ?>
                    <button class="odds-btn" onclick="addToBetSlip(<?php echo $event['id']; ?>, 'home', <?php echo $event['home_odds']; ?>)">
                        <?php echo number_format($event['home_odds'], 2); ?>
                    </button>
                    <?php endif; ?>
                    <?php if ($event['draw_odds']): ?>
                    <button class="odds-btn" onclick="addToBetSlip(<?php echo $event['id']; ?>, 'draw', <?php echo $event['draw_odds']; ?>)">
                        <?php echo number_format($event['draw_odds'], 2); ?>
                    </button>
                    <?php endif; ?>
                    <?php if ($event['away_odds']): ?>
                    <button class="odds-btn" onclick="addToBetSlip(<?php echo $event['id']; ?>, 'away', <?php echo $event['away_odds']; ?>)">
                        <?php echo number_format($event['away_odds'], 2); ?>
                    </button>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- Filters & Search -->
<section class="filters-section">
    <div class="container">
        <div class="filters-bar">
            <div class="filter-group">
                <label class="filter-label">Time:</label>
                <select class="filter-select" id="timeFilter" onchange="applyFilters()">
                    <option value="today" <?php echo $timeFilter === 'today' ? 'selected' : ''; ?>>Today</option>
                    <option value="tomorrow" <?php echo $timeFilter === 'tomorrow' ? 'selected' : ''; ?>>Tomorrow</option>
                    <option value="week" <?php echo $timeFilter === 'week' ? 'selected' : ''; ?>>This Week</option>
                    <option value="all" <?php echo $timeFilter === 'all' ? 'selected' : ''; ?>>All Time</option>
                </select>
            </div>
            
            <?php if (!empty($leagues)): ?>
            <div class="filter-group">
                <label class="filter-label">League:</label>
                <select class="filter-select" id="leagueFilter" onchange="applyFilters()">
                    <option value="all">All Leagues</option>
                    <?php foreach ($leagues as $league): ?>
                    <option value="<?php echo $league['slug']; ?>" <?php echo $selectedLeague === $league['slug'] ? 'selected' : ''; ?>>
                        <?php echo $league['name']; ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <?php endif; ?>
            
            <div class="search-group">
                <div class="search-input">
                    <i class="fas fa-search"></i>
                    <input type="text" 
                           placeholder="Search teams, leagues..." 
                           value="<?php echo htmlspecialchars($searchQuery); ?>"
                           id="searchInput"
                           onkeyup="handleSearch(event)">
                    <?php if ($searchQuery): ?>
                    <button class="clear-search" onclick="clearSearch()">
                        <i class="fas fa-times"></i>
                    </button>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="view-toggle">
                <button class="view-btn <?php echo $viewMode === 'grid' ? 'active' : ''; ?>" 
                        onclick="changeView('grid')" data-view="grid">
                    <i class="fas fa-th"></i>
                </button>
                <button class="view-btn <?php echo $viewMode === 'list' ? 'active' : ''; ?>" 
                        onclick="changeView('list')" data-view="list">
                    <i class="fas fa-list"></i>
                </button>
            </div>
        </div>
    </div>
</section>

<!-- Featured Events -->
<?php if (!empty($featuredEvents) && $selectedSport === 'all' && empty($searchQuery)): ?>
<section class="featured-events">
    <div class="container">
        <div class="section-header">
            <h2 class="section-title">
                <i class="fas fa-star"></i>
                Featured Events
            </h2>
            <a href="#upcoming-events" class="view-all-btn">View All Events</a>
        </div>
        
        <div class="featured-grid">
            <?php foreach ($featuredEvents as $event): ?>
            <div class="featured-event-card" data-event-id="<?php echo $event['id']; ?>">
                <div class="event-header">
                    <div class="league-info">
                        <img src="<?php echo $event['league_logo'] ?: '/assets/images/default-league.png'; ?>" 
                             alt="<?php echo $event['league_name']; ?>" class="league-logo">
                        <span class="league-name"><?php echo $event['league_name']; ?></span>
                    </div>
                    <div class="event-time">
                        <?php echo formatEventDate($event['start_time']); ?>
                    </div>
                </div>
                
                <div class="match-details">
                    <div class="team home-team">
                        <img src="<?php echo $event['home_team_logo'] ?: '/assets/images/default-team.png'; ?>" 
                             alt="<?php echo $event['home_team']; ?>" class="team-logo">
                        <span class="team-name"><?php echo $event['home_team']; ?></span>
                    </div>
                    
                    <div class="vs-section">
                        <span class="vs-text">VS</span>
                        <?php if ($event['status'] === 'live'): ?>
                        <div class="live-score">
                            <span class="score"><?php echo $event['home_score']; ?>-<?php echo $event['away_score']; ?></span>
                            <span class="live-indicator">LIVE</span>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="team away-team">
                        <img src="<?php echo $event['away_team_logo'] ?: '/assets/images/default-team.png'; ?>" 
                             alt="<?php echo $event['away_team']; ?>" class="team-logo">
                        <span class="team-name"><?php echo $event['away_team']; ?></span>
                    </div>
                </div>
                
                <div class="betting-options">
                    <?php if ($event['home_odds']): ?>
                    <button class="odds-btn home" 
                            onclick="addToBetSlip(<?php echo $event['id']; ?>, 'home', <?php echo $event['home_odds']; ?>, '<?php echo addslashes($event['home_team']); ?>')">
                        <span class="selection">1</span>
                        <span class="odds"><?php echo number_format($event['home_odds'], 2); ?></span>
                    </button>
                    <?php endif; ?>
                    
                    <?php if ($event['draw_odds']): ?>
                    <button class="odds-btn draw" 
                            onclick="addToBetSlip(<?php echo $event['id']; ?>, 'draw', <?php echo $event['draw_odds']; ?>, 'Draw')">
                        <span class="selection">X</span>
                        <span class="odds"><?php echo number_format($event['draw_odds'], 2); ?></span>
                    </button>
                    <?php endif; ?>
                    
                    <?php if ($event['away_odds']): ?>
                    <button class="odds-btn away" 
                            onclick="addToBetSlip(<?php echo $event['id']; ?>, 'away', <?php echo $event['away_odds']; ?>, '<?php echo addslashes($event['away_team']); ?>')">
                        <span class="selection">2</span>
                        <span class="odds"><?php echo number_format($event['away_odds'], 2); ?></span>
                    </button>
                    <?php endif; ?>
                </div>
                
                <div class="event-actions">
                    <button class="action-btn favorite" onclick="toggleFavorite(<?php echo $event['id']; ?>)">
                        <i class="far fa-star"></i>
                    </button>
                    <button class="action-btn more-markets" onclick="showMoreMarkets(<?php echo $event['id']; ?>)">
                        <span>+<?php echo $event['total_markets']; ?> Markets</span>
                    </button>
                    <?php if ($event['live_streaming']): ?>
                    <button class="action-btn stream" onclick="openStream(<?php echo $event['id']; ?>)">
                        <i class="fas fa-play"></i>
                    </button>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- Upcoming Events -->
<section class="upcoming-events" id="upcoming-events">
    <div class="container">
        <div class="section-header">
            <h2 class="section-title">
                <i class="fas fa-calendar"></i>
                <?php 
                if ($selectedSport !== 'all') {
                    $sportName = array_filter($sportsData, fn($s) => $s['slug'] === $selectedSport)[0]['name'] ?? 'Sports';
                    echo $sportName . ' Events';
                } else {
                    echo 'Upcoming Events';
                }
                ?>
                <?php if ($searchQuery): ?>
                <span class="search-results">- Results for "<?php echo htmlspecialchars($searchQuery); ?>"</span>
                <?php endif; ?>
            </h2>
            <div class="results-count">
                <?php echo count($upcomingEvents); ?> events found
            </div>
        </div>
        
        <?php if (empty($upcomingEvents)): ?>
        <div class="empty-state">
            <div class="empty-icon">
                <i class="fas fa-calendar-times"></i>
            </div>
            <h3>No Events Found</h3>
            <p>
                <?php if ($searchQuery): ?>
                    No events match your search criteria. Try adjusting your search terms.
                <?php else: ?>
                    No events available for the selected filters. Try changing your selection.
                <?php endif; ?>
            </p>
            <div class="empty-actions">
                <?php if ($searchQuery): ?>
                <button class="btn btn-primary" onclick="clearSearch()">Clear Search</button>
                <?php endif; ?>
                <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/sportsbook.php" class="btn btn-outline">View All Sports</a>
            </div>
        </div>
        <?php else: ?>
        <div class="events-container <?php echo $viewMode; ?>-view" id="eventsContainer">
            <?php foreach ($upcomingEvents as $event): ?>
            <div class="event-card" data-event-id="<?php echo $event['id']; ?>">
                <div class="event-info">
                    <div class="event-meta">
                        <div class="sport-league">
                            <i class="<?php echo $event['sport_icon']; ?>"></i>
                            <span class="sport-name"><?php echo $event['sport_name']; ?></span>
                            <span class="separator">•</span>
                            <span class="league-name"><?php echo $event['league_name']; ?></span>
                        </div>
                        <div class="event-time">
                            <?php echo formatEventDate($event['start_time']); ?>
                        </div>
                    </div>
                    
                    <div class="match-info">
                        <div class="teams">
                            <div class="team home">
                                <img src="<?php echo $event['home_team_logo'] ?: '/assets/images/default-team.png'; ?>" 
                                     alt="<?php echo $event['home_team']; ?>" class="team-logo">
                                <span class="team-name"><?php echo $event['home_team']; ?></span>
                            </div>
                            
                            <div class="vs">
                                <?php if ($event['status'] === 'live'): ?>
                                <div class="live-info">
                                    <span class="live-badge">LIVE</span>
                                    <span class="score"><?php echo $event['home_score']; ?>-<?php echo $event['away_score']; ?></span>
                                    <span class="time"><?php echo $event['match_time']; ?>'</span>
                                </div>
                                <?php else: ?>
                                <span class="vs-text">VS</span>
                                <?php endif; ?>
                            </div>
                            
                            <div class="team away">
                                <img src="<?php echo $event['away_team_logo'] ?: '/assets/images/default-team.png'; ?>" 
                                     alt="<?php echo $event['away_team']; ?>" class="team-logo">
                                <span class="team-name"><?php echo $event['away_team']; ?></span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="betting-section">
                    <div class="main-odds">
                        <?php if ($event['home_odds']): ?>
                        <button class="odds-btn home" 
                                onclick="addToBetSlip(<?php echo $event['id']; ?>, 'home', <?php echo $event['home_odds']; ?>, '<?php echo addslashes($event['home_team']); ?>')">
                            <span class="label">1</span>
                            <span class="odds"><?php echo number_format($event['home_odds'], 2); ?></span>
                        </button>
                        <?php endif; ?>
                        
                        <?php if ($event['draw_odds']): ?>
                        <button class="odds-btn draw" 
                                onclick="addToBetSlip(<?php echo $event['id']; ?>, 'draw', <?php echo $event['draw_odds']; ?>, 'Draw')">
                            <span class="label">X</span>
                            <span class="odds"><?php echo number_format($event['draw_odds'], 2); ?></span>
                        </button>
                        <?php endif; ?>
                        
                        <?php if ($event['away_odds']): ?>
                        <button class="odds-btn away" 
                                onclick="addToBetSlip(<?php echo $event['id']; ?>, 'away', <?php echo $event['away_odds']; ?>, '<?php echo addslashes($event['away_team']); ?>')">
                            <span class="label">2</span>
                            <span class="odds"><?php echo number_format($event['away_odds'], 2); ?></span>
                        </button>
                        <?php endif; ?>
                    </div>
                    
                    <div class="event-actions">
                        <button class="action-btn favorite" onclick="toggleFavorite(<?php echo $event['id']; ?>)" title="Add to Favorites">
                            <i class="far fa-star"></i>
                        </button>
                        <button class="action-btn markets" onclick="showEventDetails(<?php echo $event['id']; ?>)" title="All Markets">
                            <span>+<?php echo $event['total_markets']; ?></span>
                        </button>
                        <?php if ($event['live_streaming']): ?>
                        <button class="action-btn stream" onclick="openStream(<?php echo $event['id']; ?>)" title="Live Stream">
                            <i class="fas fa-play"></i>
                        </button>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        
        <!-- Load More Button -->
        <div class="load-more-section">
            <button class="btn btn-outline load-more-btn" onclick="loadMoreEvents()">
                <i class="fas fa-plus"></i>
                Load More Events
            </button>
        </div>
        <?php endif; ?>
    </div>
</section>

<!-- Event Details Modal -->
<div class="modal-backdrop" id="eventDetailsModal">
    <div class="modal event-modal">
        <div class="modal-header">
            <h3 class="modal-title" id="eventModalTitle">Event Details</h3>
            <button class="modal-close" onclick="closeModal('eventDetailsModal')">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body" id="eventModalBody">
            <!-- Event details will be loaded here -->
        </div>
    </div>
</div>

<!-- Live Stream Modal -->
<div class="modal-backdrop" id="streamModal">
    <div class="modal stream-modal">
        <div class="modal-header">
            <h3 class="modal-title">Live Stream</h3>
            <button class="modal-close" onclick="closeModal('streamModal')">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body">
            <div class="stream-container" id="streamContainer">
                <!-- Stream will be loaded here -->
            </div>
        </div>
    </div>
</div>

<script>
// Page-specific JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Initialize sportsbook functionality
    initializeSportsbook();
    
    // Auto-refresh live events
    setInterval(refreshLiveEvents, 30000); // Every 30 seconds
    
    // Initialize filters
    initializeFilters();
});

function initializeSportsbook() {
    // Initialize live ticker
    initializeLiveTicker();
    
    // Initialize event cards
    initializeEventCards();
    
    // Initialize search
    initializeSearch();
}

function applyFilters() {
    const timeFilter = document.getElementById('timeFilter').value;
    const leagueFilter = document.getElementById('leagueFilter')?.value || 'all';
    const searchQuery = document.getElementById('searchInput').value;
    
    const params = new URLSearchParams(window.location.search);
    params.set('time', timeFilter);
    params.set('league', leagueFilter);
    if (searchQuery) {
        params.set('search', searchQuery);
    } else {
        params.delete('search');
    }
    
    window.location.href = `${window.location.pathname}?${params.toString()}`;
}

function handleSearch(event) {
    if (event.key === 'Enter') {
        applyFilters();
    }
}

function clearSearch() {
    document.getElementById('searchInput').value = '';
    applyFilters();
}

function changeView(viewMode) {
    const params = new URLSearchParams(window.location.search);
    params.set('view', viewMode);
    window.location.href = `${window.location.pathname}?${params.toString()}`;
}

function showEventDetails(eventId) {
    // Load event details via AJAX
    Crashify.api.request(`/api/events/${eventId}/details`)
        .then(response => {
            if (response.success) {
                document.getElementById('eventModalBody').innerHTML = response.html;
                openModal('eventDetailsModal');
            }
        })
        .catch(error => {
            Crashify.ui.toast.error('Failed to load event details');
        });
}

function refreshLiveEvents() {
    // Refresh live events data
    Crashify.api.request('/api/events/live')
        .then(response => {
            if (response.success) {
                updateLiveEvents(response.events);
            }
        })
        .catch(error => {
            console.error('Failed to refresh live events:', error);
        });
}
</script>

<?php include '../includes/footer.php'; ?>
