<?php
// pages/register.php - Registration Page
// صفحه ثبت نام

require_once '../includes/functions.php';

$page_title = 'ثبت نام';
$current_page = 'register';

// Redirect if already logged in
if (isLoggedIn()) {
    header('Location: /pages/dashboard.php');
    exit;
}

$error_message = '';
$success_message = '';
$form_data = [];

// Handle registration form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $form_data = [
        'first_name' => sanitizeInput($_POST['first_name']),
        'last_name' => sanitizeInput($_POST['last_name']),
        'email' => sanitizeInput($_POST['email']),
        'phone' => sanitizeInput($_POST['phone']),
        'password' => $_POST['password'],
        'confirm_password' => $_POST['confirm_password'],
        'birth_date' => sanitizeInput($_POST['birth_date']),
        'country' => sanitizeInput($_POST['country']),
        'currency' => sanitizeInput($_POST['currency']),
        'referral_code' => sanitizeInput($_POST['referral_code']),
        'terms_accepted' => isset($_POST['terms_accepted']),
        'newsletter' => isset($_POST['newsletter'])
    ];
    
    // Validation
    $errors = [];
    
    if (empty($form_data['first_name'])) {
        $errors[] = 'نام الزامی است';
    }
    
    if (empty($form_data['last_name'])) {
        $errors[] = 'نام خانوادگی الزامی است';
    }
    
    if (empty($form_data['email'])) {
        $errors[] = 'ایمیل الزامی است';
    } elseif (!filter_var($form_data['email'], FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'ایمیل نامعتبر است';
    }
    
    if (empty($form_data['password'])) {
        $errors[] = 'رمز عبور الزامی است';
    } elseif (strlen($form_data['password']) < 6) {
        $errors[] = 'رمز عبور باید حداقل 6 کاراکتر باشد';
    }
    
    if ($form_data['password'] !== $form_data['confirm_password']) {
        $errors[] = 'تکرار رمز عبور مطابقت ندارد';
    }
    
    if (empty($form_data['birth_date'])) {
        $errors[] = 'تاریخ تولد الزامی است';
    } else {
        $birth_date = new DateTime($form_data['birth_date']);
        $today = new DateTime();
        $age = $today->diff($birth_date)->y;
        
        if ($age < 18) {
            $errors[] = 'سن شما باید حداقل 18 سال باشد';
        }
    }
    
    if (!$form_data['terms_accepted']) {
        $errors[] = 'پذیرش قوانین و مقررات الزامی است';
    }
    
    // Check if email already exists
    if (empty($errors)) {
        $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$form_data['email']]);
        
        if ($stmt->rowCount() > 0) {
            $errors[] = 'این ایمیل قبلاً ثبت شده است';
        }
    }
    
    // Check referral code if provided
    $referrer_id = null;
    if (!empty($form_data['referral_code'])) {
        $stmt = $pdo->prepare("SELECT id FROM users WHERE referral_code = ?");
        $stmt->execute([$form_data['referral_code']]);
        $referrer = $stmt->fetch();
        
        if (!$referrer) {
            $errors[] = 'کد معرف نامعتبر است';
        } else {
            $referrer_id = $referrer['id'];
        }
    }
    
    if (empty($errors)) {
        try {
            $pdo->beginTransaction();
            
            // Generate unique referral code for new user
            $new_referral_code = generateReferralCode();
            
            // Hash password
            $hashed_password = password_hash($form_data['password'], PASSWORD_DEFAULT);
            
            // Insert new user
            $stmt = $pdo->prepare("
                INSERT INTO users (
                    first_name, last_name, email, phone, password, birth_date, 
                    country, currency, referral_code, referred_by, 
                    email_verified, status, created_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 0, 'active', NOW())
            ");
            
            $stmt->execute([
                $form_data['first_name'],
                $form_data['last_name'],
                $form_data['email'],
                $form_data['phone'],
                $hashed_password,
                $form_data['birth_date'],
                $form_data['country'],
                $form_data['currency'],
                $new_referral_code,
                $referrer_id
            ]);
            
            $user_id = $pdo->lastInsertId();
            
            // Generate email verification token
            $verification_token = bin2hex(random_bytes(32));
            $stmt = $pdo->prepare("
                INSERT INTO email_verifications (user_id, token, expires_at, created_at)
                VALUES (?, ?, DATE_ADD(NOW(), INTERVAL 24 HOUR), NOW())
            ");
            $stmt->execute([$user_id, $verification_token]);
            
            // Add welcome bonus if available
            $stmt = $pdo->prepare("
                SELECT * FROM bonuses 
                WHERE type = 'welcome' AND is_active = 1 
                AND start_date <= NOW() AND end_date >= NOW()
                LIMIT 1
            ");
            $stmt->execute();
            $welcome_bonus = $stmt->fetch();
            
            if ($welcome_bonus) {
                $bonus_amount = $welcome_bonus['amount'];
                
                // Add bonus to user
                $stmt = $pdo->prepare("
                    INSERT INTO user_bonuses (user_id, bonus_id, amount, status, claimed_at, expires_at)
                    VALUES (?, ?, ?, 'active', NOW(), DATE_ADD(NOW(), INTERVAL ? DAY))
                ");
                $stmt->execute([$user_id, $welcome_bonus['id'], $bonus_amount, $welcome_bonus['validity_days'] ?: 30]);
                
                // Update user balance
                $stmt = $pdo->prepare("UPDATE users SET balance = balance + ?, bonus_balance = bonus_balance + ? WHERE id = ?");
                $stmt->execute([$bonus_amount, $bonus_amount, $user_id]);
                
                // Create transaction record
                $stmt = $pdo->prepare("
                    INSERT INTO transactions (user_id, type, amount, status, description, created_at)
                    VALUES (?, 'bonus', ?, 'completed', 'جایزه خوش‌آمدگویی', NOW())
                ");
                $stmt->execute([$user_id, $bonus_amount]);
            }
            
            // Handle referral bonus
            if ($referrer_id) {
                $stmt = $pdo->prepare("
                    SELECT * FROM bonuses 
                    WHERE type = 'referral' AND is_active = 1 
                    AND start_date <= NOW() AND end_date >= NOW()
                    LIMIT 1
                ");
                $stmt->execute();
                $referral_bonus = $stmt->fetch();
                
                if ($referral_bonus) {
                    $referral_amount = $referral_bonus['amount'];
                    
                    // Add bonus to referrer
                    $stmt = $pdo->prepare("
                        INSERT INTO user_bonuses (user_id, bonus_id, amount, status, claimed_at, expires_at)
                        VALUES (?, ?, ?, 'active', NOW(), DATE_ADD(NOW(), INTERVAL ? DAY))
                    ");
                    $stmt->execute([$referrer_id, $referral_bonus['id'], $referral_amount, $referral_bonus['validity_days'] ?: 30]);
                    
                    // Update referrer balance
                    $stmt = $pdo->prepare("UPDATE users SET balance = balance + ?, bonus_balance = bonus_balance + ? WHERE id = ?");
                    $stmt->execute([$referral_amount, $referral_amount, $referrer_id]);
                    
                    // Create transaction record
                    $stmt = $pdo->prepare("
                        INSERT INTO transactions (user_id, type, amount, status, description, created_at)
                        VALUES (?, 'bonus', ?, 'completed', 'جایزه معرفی کاربر جدید', NOW())
                    ");
                    $stmt->execute([$referrer_id, $referral_amount]);
                }
            }
            
            // Log registration activity
            $stmt = $pdo->prepare("
                INSERT INTO user_activity (user_id, activity_type, description, ip_address, user_agent, created_at)
                VALUES (?, 'register', 'User registered', ?, ?, NOW())
            ");
            $stmt->execute([$user_id, $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT']]);
            
            // Subscribe to newsletter if requested
            if ($form_data['newsletter']) {
                $stmt = $pdo->prepare("
                    INSERT INTO newsletter_subscribers (email, user_id, status, subscribed_at)
                    VALUES (?, ?, 'active', NOW())
                ");
                $stmt->execute([$form_data['email'], $user_id]);
            }
            
            $pdo->commit();
            
            // Send verification email (placeholder)
            // sendVerificationEmail($form_data['email'], $verification_token);
            
            $success_message = 'ثبت نام با موفقیت انجام شد! لطفاً ایمیل خود را برای تأیید حساب بررسی کنید.';
            
            // Clear form data
            $form_data = [];
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $error_message = 'خطا در ثبت نام. لطفاً دوباره تلاش کنید.';
        }
    } else {
        $error_message = implode('<br>', $errors);
    }
}

// Get countries list
$countries = [
    'IR' => 'ایران',
    'US' => 'آمریکا',
    'GB' => 'انگلستان',
    'DE' => 'آلمان',
    'FR' => 'فرانسه',
    'CA' => 'کانادا',
    'AU' => 'استرالیا',
    'TR' => 'ترکیه',
    'AE' => 'امارات'
];

// Get currencies list
$currencies = [
    'USD' => 'دلار آمریکا (USD)',
    'EUR' => 'یورو (EUR)',
    'GBP' => 'پوند انگلیس (GBP)',
    'IRR' => 'ریال ایران (IRR)'
];

include '../includes/header.php';
?>

<div class="auth-container">
    <div class="auth-wrapper register-wrapper">
        <div class="auth-card register-card">
            <div class="auth-header">
                <div class="auth-logo">
                    <img src="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'assets/images/logo.png" alt="Crashify" class="logo-img">
                    <h2>Crashify</h2>
                </div>
                <h3>ایجاد حساب کاربری</h3>
                <p class="text-muted">به خانواده بزرگ Crashify بپیوندید</p>
            </div>
            
            <div class="auth-body">
                <?php if ($error_message): ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-circle"></i>
                        <?php echo $error_message; ?>
                    </div>
                <?php endif; ?>
                
                <?php if ($success_message): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle"></i>
                        <?php echo $success_message; ?>
                        <div class="mt-2">
                            <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'auth/login.php" class="btn btn-sm btn-outline-success">
                                ورود به حساب کاربری
                            </a>
                        </div>
                    </div>
                <?php endif; ?>
                
                <form method="POST" class="auth-form register-form" id="registerForm">
                    <!-- Personal Information -->
                    <div class="form-section">
                        <h5><i class="fas fa-user"></i> اطلاعات شخصی</h5>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="first_name">نام *</label>
                                    <input type="text" class="form-control" id="first_name" name="first_name" 
                                           value="<?php echo htmlspecialchars($form_data['first_name'] ?? ''); ?>" 
                                           placeholder="نام خود را وارد کنید" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="last_name">نام خانوادگی *</label>
                                    <input type="text" class="form-control" id="last_name" name="last_name" 
                                           value="<?php echo htmlspecialchars($form_data['last_name'] ?? ''); ?>" 
                                           placeholder="نام خانوادگی خود را وارد کنید" required>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="email">ایمیل *</label>
                                    <input type="email" class="form-control" id="email" name="email" 
                                           value="<?php echo htmlspecialchars($form_data['email'] ?? ''); ?>" 
                                           placeholder="ایمیل خود را وارد کنید" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="phone">شماره تلفن</label>
                                    <input type="tel" class="form-control" id="phone" name="phone" 
                                           value="<?php echo htmlspecialchars($form_data['phone'] ?? ''); ?>" 
                                           placeholder="شماره تلفن خود را وارد کنید">
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="birth_date">تاریخ تولد *</label>
                                    <input type="date" class="form-control" id="birth_date" name="birth_date" 
                                           value="<?php echo htmlspecialchars($form_data['birth_date'] ?? ''); ?>" 
                                           max="<?php echo date('Y-m-d', strtotime('-18 years')); ?>" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="country">کشور</label>
                                    <select class="form-control" id="country" name="country">
                                        <option value="">انتخاب کنید</option>
                                        <?php foreach ($countries as $code => $name): ?>
                                            <option value="<?php echo $code; ?>" 
                                                    <?php echo ($form_data['country'] ?? '') === $code ? 'selected' : ''; ?>>
                                                <?php echo $name; ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Account Information -->
                    <div class="form-section">
                        <h5><i class="fas fa-lock"></i> اطلاعات حساب</h5>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="password">رمز عبور *</label>
                                    <div class="password-input">
                                        <input type="password" class="form-control" id="password" name="password" 
                                               placeholder="رمز عبور خود را وارد کنید" required minlength="6">
                                        <button type="button" class="password-toggle" onclick="togglePassword('password')">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                    </div>
                                    <small class="form-text text-muted">حداقل 6 کاراکتر</small>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="confirm_password">تکرار رمز عبور *</label>
                                    <div class="password-input">
                                        <input type="password" class="form-control" id="confirm_password" name="confirm_password" 
                                               placeholder="رمز عبور را دوباره وارد کنید" required minlength="6">
                                        <button type="button" class="password-toggle" onclick="togglePassword('confirm_password')">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="currency">واحد پول</label>
                                    <select class="form-control" id="currency" name="currency">
                                        <?php foreach ($currencies as $code => $name): ?>
                                            <option value="<?php echo $code; ?>" 
                                                    <?php echo ($form_data['currency'] ?? 'USD') === $code ? 'selected' : ''; ?>>
                                                <?php echo $name; ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="referral_code">کد معرف (اختیاری)</label>
                                    <input type="text" class="form-control" id="referral_code" name="referral_code" 
                                           value="<?php echo htmlspecialchars($form_data['referral_code'] ?? $_GET['ref'] ?? ''); ?>" 
                                           placeholder="کد معرف را وارد کنید">
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Terms and Options -->
                    <div class="form-section">
                        <div class="form-options">
                            <div class="custom-control custom-checkbox">
                                <input type="checkbox" class="custom-control-input" id="terms_accepted" name="terms_accepted" required>
                                <label class="custom-control-label" for="terms_accepted">
                                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/terms.php" target="_blank">قوانین و مقررات</a> و 
                                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/privacy.php" target="_blank">حریم خصوصی</a> را مطالعه کرده و می‌پذیرم *
                                </label>
                            </div>
                            
                            <div class="custom-control custom-checkbox">
                                <input type="checkbox" class="custom-control-input" id="newsletter" name="newsletter" 
                                       <?php echo ($form_data['newsletter'] ?? false) ? 'checked' : ''; ?>>
                                <label class="custom-control-label" for="newsletter">
                                    عضویت در خبرنامه و دریافت اطلاعات جدید
                                </label>
                            </div>
                            
                            <div class="custom-control custom-checkbox">
                                <input type="checkbox" class="custom-control-input" id="age_confirm" required>
                                <label class="custom-control-label" for="age_confirm">
                                    تأیید می‌کنم که بالای 18 سال سن دارم *
                                </label>
                            </div>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn btn-primary btn-block auth-submit-btn">
                        <i class="fas fa-user-plus"></i>
                        ایجاد حساب کاربری
                    </button>
                </form>
                
                <!-- Social Registration -->
                <div class="social-login">
                    <div class="divider">
                        <span>یا</span>
                    </div>
                    
                    <div class="social-buttons">
                        <button class="btn btn-social btn-google" onclick="socialRegister('google')">
                            <i class="fab fa-google"></i>
                            ثبت نام با گوگل
                        </button>
                        
                        <button class="btn btn-social btn-facebook" onclick="socialRegister('facebook')">
                            <i class="fab fa-facebook-f"></i>
                            ثبت نام با فیسبوک
                        </button>
                    </div>
                </div>
            </div>
            
            <div class="auth-footer">
                <p>
                    قبلاً حساب کاربری دارید؟
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'auth/login.php" class="login-link">
                        <strong>وارد شوید</strong>
                    </a>
                </p>
            </div>
        </div>
        
        <!-- Benefits Sidebar -->
        <div class="benefits-sidebar">
            <div class="benefits-content">
                <h4>مزایای عضویت</h4>
                
                <div class="benefit-item">
                    <div class="benefit-icon">
                        <i class="fas fa-gift"></i>
                    </div>
                    <div class="benefit-text">
                        <h6>جایزه خوش‌آمدگویی</h6>
                        <p>$50 جایزه برای کاربران جدید</p>
                    </div>
                </div>
                
                <div class="benefit-item">
                    <div class="benefit-icon">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="benefit-text">
                        <h6>برنامه معرفی</h6>
                        <p>از معرفی دوستان درآمد کسب کنید</p>
                    </div>
                </div>
                
                <div class="benefit-item">
                    <div class="benefit-icon">
                        <i class="fas fa-crown"></i>
                    </div>
                    <div class="benefit-text">
                        <h6>برنامه وفاداری</h6>
                        <p>جوایز ویژه برای کاربران فعال</p>
                    </div>
                </div>
                
                <div class="benefit-item">
                    <div class="benefit-icon">
                        <i class="fas fa-mobile-alt"></i>
                    </div>
                    <div class="benefit-text">
                        <h6>دسترسی موبایل</h6>
                        <p>بازی در هر زمان و مکان</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.register-wrapper {
    max-width: 1400px;
}

.register-card {
    max-width: 800px;
}

.form-section {
    margin-bottom: 2rem;
    padding-bottom: 1.5rem;
    border-bottom: 1px solid #e9ecef;
}

.form-section:last-of-type {
    border-bottom: none;
}

.form-section h5 {
    color: #495057;
    margin-bottom: 1.5rem;
    font-weight: 600;
    display: flex;
    align-items: center;
}

.form-section h5 i {
    margin-right: 0.5rem;
    color: #667eea;
}

.form-options {
    margin-bottom: 2rem;
}

.form-options .custom-control {
    margin-bottom: 1rem;
}

.form-options .custom-control-label {
    line-height: 1.5;
}

.form-options .custom-control-label a {
    color: #667eea;
    text-decoration: none;
}

.form-options .custom-control-label a:hover {
    text-decoration: underline;
}

.benefits-sidebar {
    flex: 1;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 3rem;
    display: flex;
    align-items: center;
}

.benefits-content h4 {
    margin-bottom: 2rem;
    font-weight: bold;
    text-align: center;
}

.benefit-item {
    display: flex;
    align-items: center;
    margin-bottom: 2rem;
}

.benefit-icon {
    width: 60px;
    height: 60px;
    background: rgba(255,255,255,0.1);
    border-radius: 15px;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-right: 1rem;
    font-size: 1.5rem;
}

.benefit-text h6 {
    margin-bottom: 0.25rem;
    font-weight: 600;
}

.benefit-text p {
    margin: 0;
    opacity: 0.8;
    font-size: 0.9rem;
}

.login-link {
    color: #667eea;
    text-decoration: none;
}

.login-link:hover {
    text-decoration: underline;
}

@media (max-width: 768px) {
    .register-wrapper {
        flex-direction: column;
    }
    
    .benefits-sidebar {
        order: -1;
        padding: 2rem;
    }
    
    .benefits-content {
        text-align: center;
    }
    
    .benefit-item {
        flex-direction: column;
        text-align: center;
    }
    
    .benefit-icon {
        margin-right: 0;
        margin-bottom: 1rem;
    }
    
    .register-card {
        padding: 2rem;
    }
    
    .form-section h5 {
        font-size: 1.1rem;
    }
}
</style>

<script>
// Toggle password visibility
function togglePassword(fieldId) {
    const field = document.getElementById(fieldId);
    const toggle = field.nextElementSibling.querySelector('i');
    
    if (field.type === 'password') {
        field.type = 'text';
        toggle.classList.remove('fa-eye');
        toggle.classList.add('fa-eye-slash');
    } else {
        field.type = 'password';
        toggle.classList.remove('fa-eye-slash');
        toggle.classList.add('fa-eye');
    }
}

// Social registration (placeholder)
function socialRegister(provider) {
    alert(`ثبت نام با ${provider} به زودی فعال خواهد شد`);
}

// Form validation
document.getElementById('registerForm').addEventListener('submit', function(e) {
    const password = document.getElementById('password').value;
    const confirmPassword = document.getElementById('confirm_password').value;
    const termsAccepted = document.getElementById('terms_accepted').checked;
    const ageConfirm = document.getElementById('age_confirm').checked;
    
    if (password !== confirmPassword) {
        e.preventDefault();
        alert('تکرار رمز عبور مطابقت ندارد');
        return;
    }
    
    if (!termsAccepted) {
        e.preventDefault();
        alert('لطفاً قوانین و مقررات را بپذیرید');
        return;
    }
    
    if (!ageConfirm) {
        e.preventDefault();
        alert('لطفاً تأیید کنید که بالای 18 سال سن دارید');
        return;
    }
    
    // Show loading state
    const submitBtn = this.querySelector('.auth-submit-btn');
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> در حال ثبت نام...';
});

// Password strength indicator
document.getElementById('password').addEventListener('input', function() {
    const password = this.value;
    const strength = calculatePasswordStrength(password);
    
    // Remove existing strength indicator
    const existingIndicator = this.parentNode.querySelector('.password-strength');
    if (existingIndicator) {
        existingIndicator.remove();
    }
    
    if (password.length > 0) {
        const indicator = document.createElement('div');
        indicator.className = `password-strength strength-${strength.level}`;
        indicator.innerHTML = `<div class="strength-bar"></div><span>${strength.text}</span>`;
        this.parentNode.appendChild(indicator);
    }
});

function calculatePasswordStrength(password) {
    let score = 0;
    
    if (password.length >= 6) score++;
    if (password.length >= 8) score++;
    if (/[a-z]/.test(password)) score++;
    if (/[A-Z]/.test(password)) score++;
    if (/[0-9]/.test(password)) score++;
    if (/[^A-Za-z0-9]/.test(password)) score++;
    
    if (score < 3) return { level: 'weak', text: 'ضعیف' };
    if (score < 5) return { level: 'medium', text: 'متوسط' };
    return { level: 'strong', text: 'قوی' };
}

// Real-time password confirmation
document.getElementById('confirm_password').addEventListener('input', function() {
    const password = document.getElementById('password').value;
    const confirmPassword = this.value;
    
    if (confirmPassword.length > 0) {
        if (password === confirmPassword) {
            this.classList.remove('is-invalid');
            this.classList.add('is-valid');
        } else {
            this.classList.remove('is-valid');
            this.classList.add('is-invalid');
        }
    } else {
        this.classList.remove('is-valid', 'is-invalid');
    }
});

// Generate referral code function
function generateReferralCode() {
    const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    let result = '';
    for (let i = 0; i < 8; i++) {
        result += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    return result;
}
</script>

<?php
// Helper function to generate referral code
function generateReferralCode() {
    global $pdo;
    
    do {
        $code = '';
        $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        for ($i = 0; $i < 8; $i++) {
            $code .= $chars[rand(0, strlen($chars) - 1)];
        }
        
        // Check if code already exists
        $stmt = $pdo->prepare("SELECT id FROM users WHERE referral_code = ?");
        $stmt->execute([$code]);
    } while ($stmt->rowCount() > 0);
    
    return $code;
}

include '../includes/footer.php';
?>
