<?php
// pages/dashboard.php - User Dashboard
// داشبورد کاربر

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Require authentication
requireAuth();

// Page settings
$pageTitle = 'Dashboard - ' . $currentUser['username'];
$bodyClass = 'dashboard-page';
$breadcrumbs = [
    ['title' => 'Dashboard']
];

// Get user statistics
$userStats = [
    'total_bets' => $currentUser['total_bets'] ?? 0,
    'total_wins' => $currentUser['total_wins'] ?? 0,
    'total_losses' => $currentUser['total_losses'] ?? 0,
    'win_rate' => $currentUser['win_rate'] ?? 0,
    'total_wagered' => $currentUser['total_wagered'] ?? 0,
    'total_winnings' => $currentUser['total_winnings'] ?? 0,
    'profit_loss' => ($currentUser['total_winnings'] ?? 0) - ($currentUser['total_wagered'] ?? 0)
];

// Get recent bets
$recentBets = fetchAll(
    "SELECT b.*, s.name as sport_name, s.icon as sport_icon,
            e.home_team, e.away_team, e.league_name
     FROM bets b
     LEFT JOIN events e ON b.event_id = e.id
     LEFT JOIN sports s ON e.sport_id = s.id
     WHERE b.user_id = ?
     ORDER BY b.created_at DESC
     LIMIT 10",
    [$currentUser['id']]
);

// Get active bonuses
$activeBonuses = fetchAll(
    "SELECT ub.*, b.title, b.type, b.description
     FROM user_bonuses ub
     JOIN bonuses b ON ub.bonus_id = b.id
     WHERE ub.user_id = ? AND ub.status = 'active'
     ORDER BY ub.expires_at ASC
     LIMIT 5",
    [$currentUser['id']]
);

// Get recent transactions
$recentTransactions = fetchAll(
    "SELECT * FROM transactions
     WHERE user_id = ?
     ORDER BY created_at DESC
     LIMIT 10",
    [$currentUser['id']]
);

// Get notifications
$notifications = fetchAll(
    "SELECT * FROM notifications
     WHERE user_id = ? AND read_at IS NULL
     ORDER BY created_at DESC
     LIMIT 5",
    [$currentUser['id']]
);

// Get favorite events
$favoriteEvents = fetchAll(
    "SELECT e.*, s.name as sport_name, s.icon as sport_icon
     FROM events e
     JOIN user_favorites uf ON e.id = uf.event_id
     JOIN sports s ON e.sport_id = s.id
     WHERE uf.user_id = ? AND e.status IN ('upcoming', 'live')
     ORDER BY e.start_time ASC
     LIMIT 5",
    [$currentUser['id']]
);

$additionalScripts = [
    '/assets/js/dashboard.js',
    '/assets/js/charts.js'
];

include '../includes/header.php';
?>

<!-- Dashboard Header -->
<section class="dashboard-header">
    <div class="container">
        <div class="header-content">
            <div class="user-welcome">
                <div class="user-avatar">
                    <?php if ($currentUser['avatar']): ?>
                    <img src="<?php echo $currentUser['avatar']; ?>" alt="<?php echo $currentUser['username']; ?>">
                    <?php else: ?>
                    <div class="avatar-placeholder">
                        <?php echo strtoupper(substr($currentUser['username'], 0, 2)); ?>
                    </div>
                    <?php endif; ?>
                </div>
                <div class="welcome-text">
                    <h1 class="welcome-title">Welcome back, <?php echo $currentUser['first_name'] ?: $currentUser['username']; ?>!</h1>
                    <p class="welcome-subtitle">
                        Member since <?php echo date('M Y', strtotime($currentUser['created_at'])); ?>
                        <?php if ($currentUser['vip_level']): ?>
                        • <span class="vip-badge">VIP Level <?php echo $currentUser['vip_level']; ?></span>
                        <?php endif; ?>
                    </p>
                </div>
            </div>
            
            <div class="quick-actions">
                <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/wallet.php" class="quick-action deposit">
                    <i class="fas fa-plus"></i>
                    <span>Deposit</span>
                </a>
                <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/sportsbook.php" class="quick-action bet">
                    <i class="fas fa-futbol"></i>
                    <span>Bet Now</span>
                </a>
                <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/casino.php" class="quick-action casino">
                    <i class="fas fa-dice"></i>
                    <span>Casino</span>
                </a>
            </div>
        </div>
    </div>
</section>

<!-- Balance Cards -->
<section class="balance-section">
    <div class="container">
        <div class="balance-cards">
            <div class="balance-card main">
                <div class="card-header">
                    <h3 class="card-title">
                        <i class="fas fa-wallet"></i>
                        Main Balance
                    </h3>
                    <button class="refresh-btn" onclick="refreshBalance()">
                        <i class="fas fa-sync-alt"></i>
                    </button>
                </div>
                <div class="balance-amount">
                    <span class="currency">$</span>
                    <span class="amount" id="mainBalance"><?php echo number_format($currentUser['balance'], 2); ?></span>
                </div>
                <div class="balance-actions">
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/wallet.php?action=deposit" class="btn btn-primary btn-sm">
                        <i class="fas fa-plus"></i>
                        Deposit
                    </a>
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/wallet.php?action=withdraw" class="btn btn-outline btn-sm">
                        <i class="fas fa-minus"></i>
                        Withdraw
                    </a>
                </div>
            </div>
            
            <div class="balance-card bonus">
                <div class="card-header">
                    <h3 class="card-title">
                        <i class="fas fa-gift"></i>
                        Bonus Balance
                    </h3>
                </div>
                <div class="balance-amount">
                    <span class="currency">$</span>
                    <span class="amount"><?php echo number_format($currentUser['bonus_balance'] ?? 0, 2); ?></span>
                </div>
                <div class="balance-info">
                    <span class="info-text">
                        <?php echo count($activeBonuses); ?> active bonus<?php echo count($activeBonuses) !== 1 ? 'es' : ''; ?>
                    </span>
                </div>
            </div>
            
            <div class="balance-card loyalty">
                <div class="card-header">
                    <h3 class="card-title">
                        <i class="fas fa-star"></i>
                        Loyalty Points
                    </h3>
                </div>
                <div class="balance-amount">
                    <span class="amount"><?php echo number_format($currentUser['loyalty_points'] ?? 0); ?></span>
                    <span class="unit">pts</span>
                </div>
                <div class="balance-info">
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/vip.php" class="info-link">View VIP Benefits</a>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Statistics Overview -->
<section class="stats-section">
    <div class="container">
        <div class="section-header">
            <h2 class="section-title">
                <i class="fas fa-chart-bar"></i>
                Your Statistics
            </h2>
            <div class="time-filter">
                <select id="statsTimeFilter" onchange="updateStats()">
                    <option value="7">Last 7 days</option>
                    <option value="30" selected>Last 30 days</option>
                    <option value="90">Last 90 days</option>
                    <option value="365">Last year</option>
                    <option value="all">All time</option>
                </select>
            </div>
        </div>
        
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-dice"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-value"><?php echo number_format($userStats['total_bets']); ?></div>
                    <div class="stat-label">Total Bets</div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-trophy"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-value"><?php echo number_format($userStats['total_wins']); ?></div>
                    <div class="stat-label">Wins</div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-percentage"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-value"><?php echo number_format($userStats['win_rate'], 1); ?>%</div>
                    <div class="stat-label">Win Rate</div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-coins"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-value">$<?php echo number_format($userStats['total_wagered'], 2); ?></div>
                    <div class="stat-label">Total Wagered</div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-money-bill-wave"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-value">$<?php echo number_format($userStats['total_winnings'], 2); ?></div>
                    <div class="stat-label">Total Winnings</div>
                </div>
            </div>
            
            <div class="stat-card <?php echo $userStats['profit_loss'] >= 0 ? 'positive' : 'negative'; ?>">
                <div class="stat-icon">
                    <i class="fas fa-chart-line"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-value">
                        <?php echo $userStats['profit_loss'] >= 0 ? '+' : ''; ?>$<?php echo number_format($userStats['profit_loss'], 2); ?>
                    </div>
                    <div class="stat-label">Profit/Loss</div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Main Dashboard Content -->
<div class="dashboard-content">
    <div class="container">
        <div class="dashboard-grid">
            <!-- Left Column -->
            <div class="dashboard-left">
                <!-- Recent Bets -->
                <div class="dashboard-widget">
                    <div class="widget-header">
                        <h3 class="widget-title">
                            <i class="fas fa-history"></i>
                            Recent Bets
                        </h3>
                        <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/bet-history.php" class="view-all-link">View All</a>
                    </div>
                    <div class="widget-content">
                        <?php if (empty($recentBets)): ?>
                        <div class="empty-state">
                            <i class="fas fa-dice"></i>
                            <p>No bets placed yet</p>
                            <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/sportsbook.php" class="btn btn-primary btn-sm">Place Your First Bet</a>
                        </div>
                        <?php else: ?>
                        <div class="bets-list">
                            <?php foreach ($recentBets as $bet): ?>
                            <div class="bet-item">
                                <div class="bet-info">
                                    <div class="bet-event">
                                        <i class="<?php echo $bet['sport_icon']; ?>"></i>
                                        <span class="event-name">
                                            <?php echo $bet['home_team'] . ' vs ' . $bet['away_team']; ?>
                                        </span>
                                    </div>
                                    <div class="bet-details">
                                        <span class="bet-type"><?php echo ucfirst($bet['bet_type']); ?></span>
                                        <span class="bet-odds">@<?php echo number_format($bet['odds'], 2); ?></span>
                                    </div>
                                </div>
                                <div class="bet-amount">
                                    <div class="stake">$<?php echo number_format($bet['stake'], 2); ?></div>
                                    <div class="status status-<?php echo $bet['status']; ?>">
                                        <?php echo ucfirst($bet['status']); ?>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Active Bonuses -->
                <?php if (!empty($activeBonuses)): ?>
                <div class="dashboard-widget">
                    <div class="widget-header">
                        <h3 class="widget-title">
                            <i class="fas fa-gift"></i>
                            Active Bonuses
                        </h3>
                        <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/promotions.php" class="view-all-link">View All</a>
                    </div>
                    <div class="widget-content">
                        <div class="bonuses-list">
                            <?php foreach ($activeBonuses as $bonus): ?>
                            <div class="bonus-item">
                                <div class="bonus-info">
                                    <h4 class="bonus-title"><?php echo $bonus['title']; ?></h4>
                                    <div class="bonus-amount">$<?php echo number_format($bonus['amount'], 2); ?></div>
                                </div>
                                <div class="bonus-progress">
                                    <div class="progress-bar">
                                        <div class="progress-fill" style="width: <?php echo min(100, ($bonus['wagered_amount'] / $bonus['wagering_requirement']) * 100); ?>%"></div>
                                    </div>
                                    <div class="progress-text">
                                        <?php echo number_format(($bonus['wagered_amount'] / $bonus['wagering_requirement']) * 100, 1); ?>% complete
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Right Column -->
            <div class="dashboard-right">
                <!-- Notifications -->
                <?php if (!empty($notifications)): ?>
                <div class="dashboard-widget">
                    <div class="widget-header">
                        <h3 class="widget-title">
                            <i class="fas fa-bell"></i>
                            Notifications
                            <span class="notification-count"><?php echo count($notifications); ?></span>
                        </h3>
                        <button class="mark-all-read" onclick="markAllNotificationsRead()">Mark all read</button>
                    </div>
                    <div class="widget-content">
                        <div class="notifications-list">
                            <?php foreach ($notifications as $notification): ?>
                            <div class="notification-item" data-id="<?php echo $notification['id']; ?>">
                                <div class="notification-icon">
                                    <i class="<?php echo $notification['icon'] ?? 'fas fa-info-circle'; ?>"></i>
                                </div>
                                <div class="notification-content">
                                    <h4 class="notification-title"><?php echo $notification['title']; ?></h4>
                                    <p class="notification-message"><?php echo $notification['message']; ?></p>
                                    <span class="notification-time"><?php echo timeAgo($notification['created_at']); ?></span>
                                </div>
                                <button class="notification-close" onclick="markNotificationRead(<?php echo $notification['id']; ?>)">
                                    <i class="fas fa-times"></i>
                                </button>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Recent Transactions -->
                <div class="dashboard-widget">
                    <div class="widget-header">
                        <h3 class="widget-title">
                            <i class="fas fa-exchange-alt"></i>
                            Recent Transactions
                        </h3>
                        <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/wallet.php" class="view-all-link">View All</a>
                    </div>
                    <div class="widget-content">
                        <?php if (empty($recentTransactions)): ?>
                        <div class="empty-state">
                            <i class="fas fa-receipt"></i>
                            <p>No transactions yet</p>
                        </div>
                        <?php else: ?>
                        <div class="transactions-list">
                            <?php foreach ($recentTransactions as $transaction): ?>
                            <div class="transaction-item">
                                <div class="transaction-icon">
                                    <i class="fas fa-<?php echo $transaction['type'] === 'deposit' ? 'plus' : 'minus'; ?>"></i>
                                </div>
                                <div class="transaction-info">
                                    <div class="transaction-type"><?php echo ucfirst($transaction['type']); ?></div>
                                    <div class="transaction-method"><?php echo $transaction['payment_method']; ?></div>
                                    <div class="transaction-time"><?php echo timeAgo($transaction['created_at']); ?></div>
                                </div>
                                <div class="transaction-amount">
                                    <div class="amount <?php echo $transaction['type']; ?>">
                                        <?php echo $transaction['type'] === 'deposit' ? '+' : '-'; ?>$<?php echo number_format($transaction['amount'], 2); ?>
                                    </div>
                                    <div class="status status-<?php echo $transaction['status']; ?>">
                                        <?php echo ucfirst($transaction['status']); ?>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Favorite Events -->
                <?php if (!empty($favoriteEvents)): ?>
                <div class="dashboard-widget">
                    <div class="widget-header">
                        <h3 class="widget-title">
                            <i class="fas fa-star"></i>
                            Favorite Events
                        </h3>
                    </div>
                    <div class="widget-content">
                        <div class="events-list">
                            <?php foreach ($favoriteEvents as $event): ?>
                            <div class="event-item">
                                <div class="event-info">
                                    <i class="<?php echo $event['sport_icon']; ?>"></i>
                                    <div class="event-details">
                                        <div class="event-teams"><?php echo $event['home_team'] . ' vs ' . $event['away_team']; ?></div>
                                        <div class="event-time"><?php echo formatEventDate($event['start_time']); ?></div>
                                    </div>
                                </div>
                                <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/event-details.php?id=<?php echo $event['id']; ?>" class="btn btn-outline btn-xs">
                                    Bet
                                </a>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    initializeDashboard();
});

function initializeDashboard() {
    // Initialize dashboard functionality
    updateLastActivity();
    
    // Auto-refresh balance every 30 seconds
    setInterval(refreshBalance, 30000);
    
    // Auto-refresh notifications every 60 seconds
    setInterval(refreshNotifications, 60000);
}

function refreshBalance() {
    Crashify.api.request('/api/user/balance')
        .then(response => {
            if (response.success) {
                document.getElementById('mainBalance').textContent = response.balance.toFixed(2);
            }
        })
        .catch(error => {
            console.error('Failed to refresh balance:', error);
        });
}

function updateStats() {
    const timeFilter = document.getElementById('statsTimeFilter').value;
    
    Crashify.api.request(`/api/user/stats?period=${timeFilter}`)
        .then(response => {
            if (response.success) {
                // Update stats display
                updateStatsDisplay(response.stats);
            }
        })
        .catch(error => {
            console.error('Failed to update stats:', error);
        });
}

function markNotificationRead(notificationId) {
    Crashify.api.request(`/api/notifications/${notificationId}/read`, {
        method: 'POST'
    })
    .then(response => {
        if (response.success) {
            const notificationElement = document.querySelector(`[data-id="${notificationId}"]`);
            if (notificationElement) {
                notificationElement.remove();
            }
        }
    });
}

function markAllNotificationsRead() {
    Crashify.api.request('/api/notifications/mark-all-read', {
        method: 'POST'
    })
    .then(response => {
        if (response.success) {
            const notificationsList = document.querySelector('.notifications-list');
            if (notificationsList) {
                notificationsList.innerHTML = '<div class="empty-state"><i class="fas fa-check"></i><p>All notifications read</p></div>';
            }
        }
    });
}

function refreshNotifications() {
    Crashify.api.request('/api/notifications/unread')
        .then(response => {
            if (response.success && response.notifications.length > 0) {
                // Update notifications display
                updateNotificationsDisplay(response.notifications);
            }
        })
        .catch(error => {
            console.error('Failed to refresh notifications:', error);
        });
}

function updateLastActivity() {
    // Update user's last activity timestamp
    Crashify.api.request('/api/user/activity', {
        method: 'POST'
    });
}
</script>

<?php include '../includes/footer.php'; ?>