
<?php
// api/odds.php - Odds API
// API ضرایب

require_once '../config/config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

$method = $_SERVER['REQUEST_METHOD'];
$endpoint = $_GET['endpoint'] ?? '';

try {
    switch ($endpoint) {
        case 'match':
            handleMatchOddsRequest($method);
            break;
            
        case 'live':
            handleLiveOddsRequest($method);
            break;
            
        case 'history':
            handleOddsHistoryRequest($method);
            break;
            
        case 'update':
            handleOddsUpdateRequest($method);
            break;
            
        case 'compare':
            handleOddsCompareRequest($method);
            break;
            
        default:
            throw new Exception('Invalid endpoint');
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

function handleMatchOddsRequest($method) {
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    $matchId = intval($_GET['match_id'] ?? 0);
    
    if (!$matchId) {
        throw new Exception('Match ID is required');
    }
    
    // Mock odds data - in production, this would come from odds providers
    $odds = [
        'match_id' => $matchId,
        'last_update' => date('Y-m-d H:i:s'),
        'markets' => [
            'match_result' => [
                'name' => 'Match Result',
                'options' => [
                    [
                        'id' => 'home_win',
                        'name' => 'Home Win',
                        'odds' => 1.85,
                        'change' => '+0.05',
                        'trend' => 'up'
                    ],
                    [
                        'id' => 'draw',
                        'name' => 'Draw',
                        'odds' => 3.20,
                        'change' => '-0.10',
                        'trend' => 'down'
                    ],
                    [
                        'id' => 'away_win',
                        'name' => 'Away Win',
                        'odds' => 2.10,
                        'change' => '+0.15',
                        'trend' => 'up'
                    ]
                ]
            ],
            'total_goals' => [
                'name' => 'Total Goals',
                'options' => [
                    [
                        'id' => 'over_2_5',
                        'name' => 'Over 2.5',
                        'odds' => 1.75,
                        'change' => '0.00',
                        'trend' => 'stable'
                    ],
                    [
                        'id' => 'under_2_5',
                        'name' => 'Under 2.5',
                        'odds' => 2.05,
                        'change' => '0.00',
                        'trend' => 'stable'
                    ]
                ]
            ],
            'both_teams_score' => [
                'name' => 'Both Teams to Score',
                'options' => [
                    [
                        'id' => 'yes',
                        'name' => 'Yes',
                        'odds' => 1.65,
                        'change' => '+0.05',
                        'trend' => 'up'
                    ],
                    [
                        'id' => 'no',
                        'name' => 'No',
                        'odds' => 2.25,
                        'change' => '-0.05',
                        'trend' => 'down'
                    ]
                ]
            ]
        ],
        'statistics' => [
            'total_volume' => 125000.50,
            'bet_count' => 1247,
            'most_popular' => 'home_win'
        ]
    ];
    
    echo json_encode([
        'success' => true,
        'data' => $odds
    ]);
}

function handleLiveOddsRequest($method) {
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    // Get all live matches with current odds
    $liveOdds = [
        [
            'match_id' => 2,
            'home_team' => 'Green Bay Packers',
            'away_team' => 'Chicago Bears',
            'score' => '14-7',
            'minute' => 45,
            'status' => 'live',
            'odds' => [
                'home_win' => 1.65,
                'away_win' => 2.45,
                'draw' => 3.80
            ],
            'changes' => [
                'home_win' => '-0.20',
                'away_win' => '+0.35',
                'draw' => '+0.60'
            ],
            'last_update' => date('Y-m-d H:i:s', strtotime('-30 seconds'))
        ],
        [
            'match_id' => 5,
            'home_team' => 'Arsenal',
            'away_team' => 'Chelsea',
            'score' => '1-1',
            'minute' => 67,
            'status' => 'live',
            'odds' => [
                'home_win' => 2.10,
                'away_win' => 2.05,
                'draw' => 3.20
            ],
            'changes' => [
                'home_win' => '+0.10',
                'away_win' => '-0.05',
                'draw' => '-0.30'
            ],
            'last_update' => date('Y-m-d H:i:s', strtotime('-15 seconds'))
        ]
    ];
    
    echo json_encode([
        'success' => true,
        'data' => $liveOdds,
        'timestamp' => time(),
        'next_update' => time() + 30
    ]);
}

function handleOddsHistoryRequest($method) {
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    $matchId = intval($_GET['match_id'] ?? 0);
    $market = $_GET['market'] ?? 'match_result';
    $hours = intval($_GET['hours'] ?? 24);
    
    if (!$matchId) {
        throw new Exception('Match ID is required');
    }
    
    // Mock historical odds data
    $history = [];
    $baseTime = time() - ($hours * 3600);
    
    for ($i = 0; $i < $hours; $i++) {
        $timestamp = $baseTime + ($i * 3600);
        
        // Simulate odds movement
        $homeOdds = 1.85 + (sin($i * 0.5) * 0.3) + (rand(-10, 10) / 100);
        $awayOdds = 2.10 + (cos($i * 0.3) * 0.2) + (rand(-10, 10) / 100);
        $drawOdds = 3.20 + (sin($i * 0.2) * 0.4) + (rand(-15, 15) / 100);
        
        $history[] = [
            'timestamp' => $timestamp,
            'datetime' => date('Y-m-d H:i:s', $timestamp),
            'odds' => [
                'home_win' => round($homeOdds, 2),
                'away_win' => round($awayOdds, 2),
                'draw' => round($drawOdds, 2)
            ]
        ];
    }
    
    echo json_encode([
        'success' => true,
        'data' => [
            'match_id' => $matchId,
            'market' => $market,
            'period' => $hours . ' hours',
            'history' => $history
        ]
    ]);
}

function handleOddsUpdateRequest($method) {
    if ($method !== 'POST') {
        throw new Exception('Method not allowed');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    $matchId = intval($input['match_id'] ?? 0);
    $odds = $input['odds'] ?? [];
    
    if (!$matchId || empty($odds)) {
        throw new Exception('Match ID and odds are required');
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Odds updated successfully',
        'match_id' => $matchId,
        'updated_at' => date('Y-m-d H:i:s')
    ]);
}

function handleOddsCompareRequest($method) {
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    $matchId = intval($_GET['match_id'] ?? 0);
    
    if (!$matchId) {
        throw new Exception('Match ID is required');
    }
    
    // Mock comparison data from different bookmakers
    $comparison = [
        'match_id' => $matchId,
        'bookmakers' => [
            [
                'name' => 'Crashify',
                'odds' => [
                    'home_win' => 1.85,
                    'draw' => 3.20,
                    'away_win' => 2.10
                ],
                'margin' => 4.2
            ],
            [
                'name' => 'Competitor A',
                'odds' => [
                    'home_win' => 1.82,
                    'draw' => 3.15,
                    'away_win' => 2.15
                ],
                'margin' => 4.8
            ],
            [
                'name' => 'Competitor B',
                'odds' => [
                    'home_win' => 1.88,
                    'draw' => 3.25,
                    'away_win' => 2.05
                ],
                'margin' => 3.9
            ]
        ],
        'best_odds' => [
            'home_win' => ['bookmaker' => 'Competitor B', 'odds' => 1.88],
            'draw' => ['bookmaker' => 'Competitor B', 'odds' => 3.25],
            'away_win' => ['bookmaker' => 'Competitor A', 'odds' => 2.15]
        ]
    ];
    
    echo json_encode([
        'success' => true,
        'data' => $comparison
    ]);
}
?>