// مدیریت بازی Crash
class CrashGame {
    constructor() {
        this.gameInterval = null;
        this.currentMultiplier = 1.00;
        this.gameStarted = false;
        this.currentGameId = null;
        this.targetMultiplier = null;
        this.chart = null;
        this.ctx = null;
        this.userBet = null;
        this.autoCashout = null;
        this.startTime = null;
        this.duration = 10000; // 10 ثانیه
        this.init();
    }
    
    // مقداردهی اولیه بازی
    init() {
        this.setupChart();
        this.setupEventListeners();
        this.updateUserStats();
        
        // شروع بازی اگر لازم باشد
        if (typeof currentGameId !== 'undefined' && typeof currentMultiplier !== 'undefined') {
            this.startGame(currentGameId, currentMultiplier);
        }
    }
    
    // راه‌اندازی نمودار
    setupChart() {
        this.ctx = document.getElementById('crashChart').getContext('2d');
        
        this.chart = new Chart(this.ctx, {
            type: 'line',
            data: {
                datasets: [{
                    label: 'Multiplier',
                    data: [],
                    borderColor: 'rgba(78, 204, 163, 1)',
                    borderWidth: 3,
                    pointRadius: 0,
                    fill: false,
                    tension: 0.4,
                    segment: {
                        borderColor: ctx => {
                            const value = ctx.p0.parsed.y;
                            if (value < 2) return 'rgba(78, 204, 163, 1)';
                            if (value < 5) return 'rgba(255, 204, 0, 1)';
                            return 'rgba(233, 69, 96, 1)';
                        }
                    }
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                animation: {
                    duration: 0
                },
                scales: {
                    x: {
                        type: 'linear',
                        display: false,
                        min: 0,
                        max: 10
                    },
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(255, 255, 255, 0.1)'
                        },
                        ticks: {
                            color: '#fff',
                            callback: function(value) {
                                return value + 'x';
                            }
                        }
                    }
                },
                plugins: {
                    legend: {
                        display: false
                    },
                    tooltip: {
                        enabled: false
                    },
                    annotation: {
                        annotations: {
                            line1: {
                                type: 'line',
                                yMin: 1,
                                yMax: 1,
                                borderColor: 'rgba(255, 255, 255, 0.2)',
                                borderWidth: 1,
                                borderDash: [5, 5]
                            }
                        }
                    }
                }
            }
        });
    }
    
    // راه‌اندازی گوش‌دهندگان رویداد
    setupEventListeners() {
        // دکمه ثبت شرط
        $('#placeBetBtn').click(() => {
            this.placeBet();
        });
        
        // دکمه نقد کردن
        $('#cashoutBtn').click(() => {
            this.cashout();
        });
        
        // دکمه‌های شرط سریع
        $('.quick-bet-btn').click((e) => {
            const multiplier = $(e.currentTarget).data('multiplier');
            this.applyQuickBet(multiplier);
        });
        
        // تغییر مبلغ شرط
        $('#betAmount').on('input', () => {
            this.updatePotentialProfit();
        });
        
        // تغییر نقد کردن خودکار
        $('#autoCashout').on('input', () => {
            this.autoCashout = $('#autoCashout').val() ? parseFloat($('#autoCashout').val()) : null;
        });
    }
    
    // شروع بازی
    startGame(gameId, multiplier) {
        if (this.gameStarted) return;
        
        this.gameStarted = true;
        this.currentMultiplier = 1.00;
        this.currentGameId = gameId;
        this.targetMultiplier = multiplier;
        this.startTime = Date.now();
        
        // ریست کردن نمودار
        this.chart.data.datasets[0].data = [];
        this.chart.update();
        
        // به روز رسانی نمایشگر
        $('#currentMultiplier').text('1.00x').css('color', 'var(--primary)');
        
        // فعال کردن دکمه‌ها
        $('#placeBetBtn').prop('disabled', false);
        $('#cashoutBtn').prop('disabled', true);
        
        // پنهان کردن شرط کاربر قبلی
        this.userBet = null;
        this.updateUserStats();
        
        // شروع انیمیشن بازی
        this.runGame();
        
        // اطلاع‌رسانی شروع بازی
        this.showNotification('بازی جدید شروع شد!', 'info');
    }
    
    // اجرای بازی
    runGame() {
        this.gameInterval = setInterval(() => {
            const elapsed = Date.now() - this.startTime;
            const progress = Math.min(elapsed / this.duration, 1);
            
            // محاسبه ضریب بر اساس تابع نمایی
            this.currentMultiplier = 1 + (this.targetMultiplier - 1) * progress;
            
            // به روز رسانی نمایشگر
            $('#currentMultiplier').text(this.currentMultiplier.toFixed(2) + 'x');
            
            // تغییر رنگ نمایشگر بر اساس ضریب
            if (this.currentMultiplier < 2) {
                $('#currentMultiplier').css('color', 'var(--primary)');
            } else if (this.currentMultiplier < 5) {
                $('#currentMultiplier').css('color', 'var(--warning)');
            } else {
                $('#currentMultiplier').css('color', 'var(--error)');
            }
            
            // به روز رسانی نمودار
            this.chart.data.datasets[0].data.push({
                x: elapsed / 1000,
                y: this.currentMultiplier
            });
            
            this.chart.update();
            
            // بررسی نقد کردن خودکار
            if (this.userBet && this.autoCashout && this.currentMultiplier >= this.autoCashout) {
                this.cashout();
            }
            
            // بررسی پایان بازی
            if (progress >= 1) {
                this.endGame();
            }
        }, 50);
    }
    
    // پایان بازی
    endGame() {
        clearInterval(this.gameInterval);
        this.gameStarted = false;
        
        // به روز رسانی نمایشگر
        $('#currentMultiplier').text(this.targetMultiplier.toFixed(2) + 'x').css('color', 'var(--error)');
        
        // غیرفعال کردن دکمه‌ها
        $('#placeBetBtn').prop('disabled', false);
        $('#cashoutBtn').prop('disabled', true);
        
        // اگر کاربر شرط ثبت کرده بود و نقد نکرده بود، شرط باخته محسوب می‌شود
        if (this.userBet) {
            this.showNotification(`بازی در ${this.targetMultiplier.toFixed(2)}x Crash کرد!`, 'error');
            this.userBet = null;
            this.updateUserStats();
        }
        
        // به روز رسانی تاریخچه بازی
        this.updateGameHistory();
        
        // شروع بازی جدید بعد از 5 ثانیه
        setTimeout(() => {
            this.requestNewGame();
        }, 5000);
    }
    
    // درخواست بازی جدید
    requestNewGame() {
        $.ajax({
            url: 'games/crash.php',
            type: 'POST',
            data: { action: 'start_game' },
            success: (response) => {
                if (response.success) {
                    this.startGame(response.game_id, response.multiplier);
                } else {
                    this.showNotification('خطا در شروع بازی جدید', 'error');
                }
            },
            error: () => {
                this.showNotification('خطا در ارتباط با سرور', 'error');
            }
        });
    }
    
    // ثبت شرط
    placeBet() {
        const betAmount = parseFloat($('#betAmount').val());
        
        if (!betAmount || betAmount <= 0) {
            this.showNotification('لطفا مبلغ شرط را وارد کنید', 'error');
            return;
        }
        
        if (betAmount > userBalance) {
            this.showNotification('موجودی ناکافی', 'error');
            return;
        }
        
        this.autoCashout = $('#autoCashout').val() ? parseFloat($('#autoCashout').val()) : null;
        
        $.ajax({
            url: 'games/crash.php',
            type: 'POST',
            data: {
                action: 'place_bet',
                game_id: this.currentGameId,
                amount: betAmount,
                auto_cashout: this.autoCashout
            },
            success: (response) => {
                if (response.success) {
                    this.userBet = {
                        amount: betAmount,
                        autoCashout: this.autoCashout
                    };
                    
                    this.showNotification('شرط با موفقیت ثبت شد!', 'success');
                    this.updateUserStats();
                    
                    // غیرفعال کردن دکمه ثبت شرط و فعال کردن دکمه نقد کردن
                    $('#placeBetBtn').prop('disabled', true);
                    $('#cashoutBtn').prop('disabled', false);
                } else {
                    this.showNotification(response.message, 'error');
                }
            },
            error: () => {
                this.showNotification('خطا در ثبت شرط', 'error');
            }
        });
    }
    
    // نقد کردن
    cashout() {
        if (!this.userBet) return;
        
        $.ajax({
            url: 'games/crash.php',
            type: 'POST',
            data: {
                action: 'cashout',
                game_id: this.currentGameId,
                multiplier: this.currentMultiplier
            },
            success: (response) => {
                if (response.success) {
                    this.showNotification(
                        `با موفقیت در ${this.currentMultiplier.toFixed(2)}x نقد شد! سود: $${response.win_amount.toFixed(2)}`,
                        'success'
                    );
                    
                    // به روز رسانی موجودی کاربر
                    userBalance = response.new_balance;
                    $('.user-balance').text('$' + response.new_balance.toFixed(2));
                    
                    // ریست کردن شرط کاربر
                    this.userBet = null;
                    this.updateUserStats();
                    
                    // غیرفعال کردن دکمه نقد کردن
                    $('#cashoutBtn').prop('disabled', true);
                } else {
                    this.showNotification(response.message, 'error');
                }
            },
            error: () => {
                this.showNotification('خطا در نقد کردن', 'error');
            }
        });
    }
    
    // اعمال شرط سریع
    applyQuickBet(multiplier) {
        const betAmount = parseFloat($('#betAmount').val());
        
        if (multiplier === 'max') {
            $('#betAmount').val(userBalance.toFixed(2));
        } else if (multiplier === '0.5') {
            $('#betAmount').val((betAmount * 0.5).toFixed(2));
        } else if (multiplier === '2') {
            $('#betAmount').val((betAmount * 2).toFixed(2));
        }
        
        this.updatePotentialProfit();
    }
    
    // به روز رسانی آمار کاربر
    updateUserStats() {
        if (this.userBet) {
            $('#activeBetAmount').text('$' + this.userBet.amount.toFixed(2));
            this.updatePotentialProfit();
        } else {
            $('#activeBetAmount').text('$0.00');
            $('#potentialProfit').text('$0.00');
        }
    }
    
    // به روز رسانی سود احتمالی
    updatePotentialProfit() {
        if (!this.userBet) return;
        
        const betAmount = parseFloat($('#betAmount').val());
        const potentialProfit = betAmount * this.currentMultiplier;
        $('#potentialProfit').text('$' + potentialProfit.toFixed(2));
    }
    
    // به روز رسانی تاریخچه بازی
    updateGameHistory() {
        $.ajax({
            url: 'includes/get_game_history.php',
            type: 'GET',
            success: (response) => {
                if (response.success) {
                    $('.history-list').html('');
                    
                    response.games.forEach((game) => {
                        const historyItem = `
                            <div class="history-item" data-multiplier="${game.crashed_at}">
                                <div class="game-number">#${game.id}</div>
                                <div class="multiplier-badge">
                                    <span class="multiplier">${game.crashed_at}x</span>
                                </div>
                                <div class="game-time">${game.time}</div>
                            </div>
                        `;
                        
                        $('.history-list').append(historyItem);
                    });
                }
            },
            error: () => {
                console.log('Error updating game history');
            }
        });
    }
    
    // نمایش نوتیفیکیشن
    showNotification(message, type = 'info') {
        Toastify({
            text: message,
            duration: 3000,
            gravity: "top",
            position: "right",
            backgroundColor: type === 'success' ? 'var(--success)' : 
                            type === 'error' ? 'var(--error)' : 
                            type === 'warning' ? 'var(--warning)' : 'var(--primary)',
            stopOnFocus: true
        }).showToast();
    }
}

// تابع مقداردهی اولیه برای استفاده جهانی
function initGame(gameId, multiplier) {
    window.crashGame = new CrashGame();
    window.crashGame.startGame(gameId, multiplier);
}

// بارگذاری بازیکنان فعال
function loadActivePlayers() {
    $.ajax({
        url: 'includes/get_active_players.php',
        type: 'GET',
        success: (response) => {
            if (response.success) {
                $('#activePlayers').html('');
                
                response.players.forEach((player) => {
                    const playerElement = `
                        <div class="active-player">
                            <div class="player-avatar">
                                ${player.username.charAt(0).toUpperCase()}
                            </div>
                            <div class="player-info">
                                <div class="player-name">${player.username}</div>
                                <div class="player-bet">$${player.bet_amount}</div>
                            </div>
                        </div>
                    `;
                    
                    $('#activePlayers').append(playerElement);
                });
            }
        },
        error: () => {
            console.log('Error loading active players');
        }
    });
}

// تایمر بازی
function startGameTimer() {
    // این تابع می‌تواند یک تایمر معکوس برای بازی بعدی نمایش دهد
}

// مدیریت مودال‌ها
function openDepositModal() {
    $('#depositModal').addClass('active');
}

function closeModal(modalId) {
    $('#' + modalId).removeClass('active');
}

function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(() => {
        window.crashGame.showNotification('آدرس کپی شد!', 'success');
    });
}

// تغییر تم
function toggleTheme() {
    const themes = ['default', 'dark', 'light'];
    const currentTheme = '<?php echo $_SESSION['theme'] ?? 'default'; ?>';
    const currentIndex = themes.indexOf(currentTheme);
    const nextIndex = (currentIndex + 1) % themes.length;
    const nextTheme = themes[nextIndex];
    
    $.ajax({
        url: 'includes/change_theme.php',
        type: 'POST',
        data: { theme: nextTheme },
        success: () => {
            location.reload();
        }
    });
}

// رفرش تاریخچه بازی
function refreshGameHistory() {
    if (window.crashGame) {
        window.crashGame.updateGameHistory();
        window.crashGame.showNotification('تاریخچه به روز شد', 'info');
    }
}

// مقداردهی اولیه هنگام لود صفحه
$(document).ready(function() {
    // راه‌اندازی انتخابگر زبان
    $('#languageSelector').change(function() {
        const language = $(this).val();
        
        $.ajax({
            url: 'includes/change_language.php',
            type: 'POST',
            data: { language: language },
            success: function(response) {
                if (response.success) {
                    location.reload();
                }
            }
        });
    });
    
    // بستن مودال با کلیک خارج از آن
    $('.modal').click(function(e) {
        if (e.target === this) {
            $(this).removeClass('active');
        }
    });
    
    // جلوگیری از بستن مودال با کلیک روی محتوای آن
    $('.modal-content').click(function(e) {
        e.stopPropagation();
    });
    
    // انتخاب روش پرداخت
    $('.payment-method').click(function() {
        $('.payment-method').removeClass('selected');
        $(this).addClass('selected');
    });
});